function [ild_template,ild_angles] = get_ild_template(BRIR_files,room,cfs,dB_SPL,directory)
% Create the ILD templates
% 
%   [ild_template,ild_angles] = ...
%       get_ild_template(BRIR_files,room,cfs,dB_SPL,...
%       directory)
% 
%   [ild_template,ild_angles] = get_ild_template(...)
%   calculates the ideal ILDs at frequencies cfs and for
%   angles made available by BRIR_files in room with tag
%   room. Normally room correponds to an anechoic BRIR set.
%   The ILD is calculated at the level dB_SPL (in dB SPL)
%   which scales the ILD relative to the noise floor of 0 dB
%   SPL. The templates are saved to ILD.mat in a 'Functions'
%   subdirectory of DIRECTORY, which is loaded first and the
%   gammatone filterbank parameters are compared and
%   calculations only performed if the parameters do not
%   match.

% !---
% ==========================================================
% Last changed:     $Date: 2011-09-13 17:02:31 +0100 (Tue, 13 Sep 2011) $
% Last committed:   $Revision: 285 $
% Last changed by:  $Author: mu31ch $
% ==========================================================
% !---

if exist('ild.mat','file')==2
    load ild.mat
    file = true;
else
    file = false;
    ild = struct('template',0,'angles',0,'cfs',0,'filename','');
end

numchans = length(cfs);

wrong_BRIR = ~strcmp(get_IR_filename(BRIR_files,room,0),ild.filename);

try unequal_cfs = any(round(ild.cfs) ~= round(cfs)); catch, unequal_cfs = true; end %#ok<CTCH>

if  wrong_BRIR || unequal_cfs || unique(diff(sort(BRIR_files.(['Room_' room]).angles))) ~= unique(diff(ild.angles)) || ~file

    disp('Calculating ILD template.')
    
    ild.template = zeros(numchans,length(BRIR_files.(['Room_' room]).angles));
    ild.angles = zeros(1,length(BRIR_files.(['Room_' room]).angles));

    for N = 1:length(BRIR_files.(['Room_' room]).angles)
        IR_filename = get_IR_filename(BRIR_files,room,BRIR_files.(['Room_' room]).angles(N));
        [IR_wav,fs] = wavread(IR_filename);
        brir_length = length(IR_wav);
        y = (2*rand(fs,1))-1; % 1 second noise burst
        z = zeros(2,fs+brir_length-1);

        z(1,:) = (conv(IR_wav(:,1),y));
        z(2,:) = (conv(IR_wav(:,2),y));
        z = z(:,1:fs);
        
        bm_L = zeros(numchans,fs);
        bm_R = zeros(numchans,fs);
        env_L = zeros(numchans,fs);
        env_R = zeros(numchans,fs);
        
        for i=1:numchans
            [bm,env]=gammatone(z(1,:),fs,cfs(i));
            bm_L(i,:) = bm;
            env_L(i,:) = env;
            [bm,env]=gammatone(z(2,:),fs,cfs(i));
            bm_R(i,:) = bm;
            env_R(i,:) = env;
        end
        
        amp = dB2num(dB_SPL)/max([calc_rms(bm_L,2); calc_rms(bm_R,2)]);
        % normalise to max 60 dB SPL RMS
        env_L = amp.*env_L;
        env_R = amp.*env_R;
        
        noise_floor = (dB2num(0)^3.333)^2; % 0 dB SPL based on Meddis' numeric range

        %%%% Left ear
        env_L = env_L(:,1:fs);
        env_L = calc_anfr(env_L,fs);
        eng_L = ((env_L.^3.333).^2);
        eng_L = mean(eng_L,2) + noise_floor;

        %%%% Right ear
        env_R = env_R(:,1:fs);
        env_R = calc_anfr(env_R,fs);
        eng_R = ((env_R.^3.333).^2);
        eng_R = mean(eng_R,2) + noise_floor;

        ild.template(:,N) = 10.*log10(eng_L./eng_R);
        ild.angles(N) = BRIR_files.(['Room_' room]).angles(N);
    end

    [ild.angles,IX] = sort(ild.angles);
    ild.template = ild.template(:,IX);
    
    ild_template = ild.template;
    ild_angles = ild.angles;
    ild.cfs = cfs;
    ild.filename = get_IR_filename(BRIR_files,room,0);

    save([directory filesep 'Functions' filesep 'ild.mat'],'ild')
    disp(['Done. (ILD data save to ' directory filesep 'Functions' filesep 'ild.mat)'])
else
    ild_template = ild.template;
    ild_angles = ild.angles;
end

% [EOF]
