/* Martin's precedence model.
 Called from prec_martin.m.
 
 ==========================================================
 Last changed:     $Date: 2011-09-13 17:02:31 +0100 (Tue, 13 Sep 2011) $
 Last committed:   $Revision: 285 $
 Last changed by:  $Author: mu31ch $
 ==========================================================
 */

#include "math.h"
#include "mex.h"

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
	/* ====================== DECLARATIONS ====================== */
    
	double *hc_L,*hc_R; /* pointer to hair-cell-modelled fine structure inputs */
    double *frameCount_in; /* pointer to frame count input */
    double *frame_length_in; /* pointer to frame length input */
    double *maxlag_in; /* pointer to maximum lag input */
    double *noverlap_in; /* pointer to noverlap input */
    double *inhib; /* pointer to inhibition data input */
    
    /* pointer to running cross-correlation matrix */
    double *ccg_r;
    mxArray *ccg_r_mx[1];
    
    /* pointer to temporary CCG for each frame */
    double *ccg_temp;
    mxArray *ccg_temp_mx[1];
    
    double *ccg_out; /* pointer to output CCG */
	
    /* Indices */
	mwIndex i; /* frequency */
    mwIndex j; /* frame */
    mwIndex n; /* sample */
    mwIndex m; /* lag */
    mwIndex sample; /* matrix sample */
    mwIndex index_ccg_r; /* running cross-correlation */
    mwIndex index_ccg_temp; /* temporary CCG */
    mwIndex index_ccg_r_new; /* running cross-correlation data is moved to this point */
    mwIndex index_ccg_r_old; /* running cross-correlation data is moved from this point */
    mwIndex index_ccg; /* output CCG */
    mwIndex leftn,rightn; /* left and right samples to be cross-correlated */
    mwIndex maxn; /* mximum value of n for the frame */
    mwIndex append; /* number of sample to lookahead */
	
    /* Sizes */
	mwSize numchans; /* number of frequency channels */
    mwSize frameCount; /* number of frames */
    mwSize frame_length; /* frame length */
    mwSize maxlag; /* maximum lag */
    mwSize numsamples; /* number of input samples (in time dimension only) */
    mwSize lengthCCG; /* length of cross-correlation (2*maxlag)+1 */
    mwSize noverlap; /* number of frames to overlap in calculating CCG */
    mwSize ccg_dims[3]; /* dimensions of output CCG */
    mwSize ccg_r_dims[3]; /* dimensions of running cross-correlation */
	
    /* ====================== INPUTS ====================== */
    
	/* pointer to hair-cell-modelled fine structure inputs */
	hc_L = mxGetPr(prhs[0]);
	hc_R = mxGetPr(prhs[1]);
    
	frameCount_in = mxGetPr(prhs[2]); /* pointer to frame count input */
	frame_length_in = mxGetPr(prhs[3]); /* pointer to frame length input */
	maxlag_in = mxGetPr(prhs[4]); /* pointer to maximum lag input */
	noverlap_in = mxGetPr(prhs[5]); /* pointer to noverlap input */
	inhib = mxGetPr(prhs[6]); /* pointer to inhibition data input */
    
	numsamples = mxGetM(prhs[0]); /* number of input samples (in time dimension only) */
    numchans = mxGetN(prhs[0]); /* number of frequency channels */
	
	/* ====================== DERIVE VARIABLES ====================== */
    
	frameCount = *frameCount_in; /* frame count input */
	frame_length = *frame_length_in; /* frame length input */
	maxlag = *maxlag_in; /* maximum lag */
	noverlap = *noverlap_in; /* number of frames to overlap in calculating CCG */
	lengthCCG = (2*maxlag)+1; /* length of cross-correlation */
    
    /* output CCG dimensions */
	ccg_dims[0] = lengthCCG;
	ccg_dims[1] = numchans;
	ccg_dims[2] = frameCount;
    
    /* running cross-correlation dimensions */
	ccg_r_dims[0] = lengthCCG;
	ccg_r_dims[1] = noverlap*frame_length;
	ccg_r_dims[2] = numchans;
	
	/* temp CCG */
    ccg_temp_mx[0] = mxCreateDoubleMatrix((mwSize)lengthCCG,(mwSize)numchans,mxREAL);
	ccg_temp = mxGetPr(ccg_temp_mx[0]);
    
    /* running cross-correlation */
	ccg_r_mx[0] = mxCreateNumericArray((mwSize)3,ccg_r_dims,mxDOUBLE_CLASS,mxREAL);
	ccg_r = mxGetPr(ccg_r_mx[0]);
	
    /* ====================== OUTPUT ====================== */
    
	/* CCG output */
	plhs[0] = mxCreateNumericArray(3,ccg_dims,mxDOUBLE_CLASS,mxREAL);
	ccg_out = mxGetPr(plhs[0]);
    
    /* ====================== CROSS-CORRELATE ====================== */
	
	for ( j = 0; j < frameCount; j++ ) {
		for ( i = 0; i < numchans; i++ ) {
			index_ccg = (mwIndex)(i+j*numchans)*lengthCCG;
            index_ccg_temp = (mwIndex)(i*lengthCCG);
            if ( j==0 ) { /* Calculations for first frame */
                sample = (mwIndex)(i*numsamples)+(j*frame_length);
                /* calculate for noverlap frames */
                maxn = noverlap*frame_length;
                append = 0;
            }
            else { /* Calculations for subsequent frames */
                sample = (mwIndex)(i*numsamples)+((j+noverlap-1)*frame_length);
                /* calculate for 1 frame, lookahead to noverlap^th frame */
                maxn = frame_length;
                append = (noverlap-1)*frame_length;
            }
            /* Cross-correlate */
			for ( n = 0; n < maxn; n++ ) {
                index_ccg_r = (mwIndex)(n+append+i*frame_length*noverlap)*lengthCCG;
				for ( m = 0; m < lengthCCG; m++ ) { /* calculate cross-correlations */
                    leftn = sample+n+( (m>maxlag) ? (m-maxlag) : (0) );
                    rightn = sample+n+( (m<maxlag) ? (maxlag-m) : (0) );
					ccg_r[index_ccg_r+m] = hc_L[leftn]*hc_R[rightn]*inhib[sample+n];
                    ccg_temp[index_ccg_temp+m] += ccg_r[index_ccg_r+m]; /* integrate ccgs */
				}
			}
            for ( m = 0; m < lengthCCG; m++ ) {
                ccg_out[index_ccg+m] = ccg_temp[index_ccg_temp+m]/(double)(frame_length*noverlap); /* write CCG to o/p */
                ccg_temp[index_ccg_temp+m] = 0.0; /* reset ccg_temp */
            }
            /* move ccg_r frames 2:noverlap to 1:noverlap-1 */
            for ( n = 0; n < (noverlap-1)*frame_length; n++ ) {
                index_ccg_r_new = (mwIndex)(n+i*frame_length*noverlap)*lengthCCG;
                index_ccg_r_old = (mwIndex)(n+frame_length+i*frame_length*noverlap)*lengthCCG;
                for ( m = 0; m < lengthCCG; m++ ) {
                    ccg_r[index_ccg_r_new+m] = ccg_r[index_ccg_r_old+m];
                    ccg_temp[index_ccg_temp+m] += ccg_r[index_ccg_r_new+m];
                }
            }
		} /* end frequency loop */
	}
    /* Destroy mx arrays */
	mxDestroyArray(*ccg_r_mx);
    mxDestroyArray(*ccg_temp_mx);
	return;
}
