function output = resynthesise(x,fs,cfs,m,frame_d,eta)
% Resynthesis a target from a binary mask
% 
%   output = resynthesise(x,fs,cfs,m,frame_d,eta)
% 
%   This function resynthesises a target vector from a
%   mixture vector using a binary mask.
% 
%   output = resynthesise(x,fs,cfs,eta,m,frame_d) takes an
%   input vector x, sampled at fs Hz, and applies the
%   binary mask m. The mask m is a time-frequency mask, with
%   one row for each frequency channel. The centre frequency
%   at each row is contained in cfs. The length of each
%   frame in the mask is specified in frame_d.
% 
%   output = resynthesise(...,eta) allows option spectral
%   energy normalisation to be performed, by multiplying
%   each frequency channel with the inverse of the
%   corresponding coefficient in eta.
% 
%   Algorithm
% 
%   The algorithm has the following steps:
%   1) Obtain the full binary mask. See GET_FULL_MASK.
%   2) Pass the vector x through a gammatone filterbank with
%      centre frequencies cfs (forwards and backwards to
%      align the phase).
%   3) Multiply the full mask with the output of the
%      gammatone filterbank.
%   4) Optionally perform spectral energy normalisation.
%   5) Sum the responses of the filterbank to produce a
%      vector the same size as x.
% 
%   See also GET_FULL_MASK.

% !---
% ==========================================================
% Last changed:     $Date: 2011-09-13 17:02:31 +0100 (Tue, 13 Sep 2011) $
% Last committed:   $Revision: 285 $
% Last changed by:  $Author: mu31ch $
% ==========================================================
% !---

if numel(x)~=max(size(x))
    error('x must be a vector')
end
if nargin<5
    error('Not enough input arguments')
end

numchans = size(m,1);
frameCount = size(m,2);
bm_phase_align = zeros(numchans,length(x));

if nargin<6
    eta = ones(1,numchans);
end
norm = 1./eta;
norm(isnan(norm)) = 0;

% get sample-by-sample masks
m_full = get_full_mask(m,frame_d);

% Phase correct the gammatone filter outputs
for i = 1:numchans
    [bm_phase_align(i,:),~,~,delay] = gammatone(x,fs,cfs(i),true); % phase aligned GTFB
    bm_phase_align(i,:) = norm(i).*bm_phase_align(i,:);
    m_full(i,:) = [m_full(i,delay+1:end) zeros(1,delay)]; % delay full mask to account for phase alignment
end

% crop to the frame length
bm_phase_align = bm_phase_align(:,1:frameCount*frame_d);

% apply mask
bm_phase_align = bm_phase_align.*m_full;

% Sum to create waveforms
output = sum(bm_phase_align);

% [EOF]
